////////////////////////////////////////////////////////////////////
//
//  Dec-2021 E. Nacher (after JL. Tain) --> DetectorConstruction.cc
//
//  Oversimplified geometry made of the BaF2 TAS detector
//  used for the master students lab work.
//
////////////////////////////////////////////////////////////////////


#include "DetectorConstruction.hh"

#include "G4Material.hh"
#include "G4Tubs.hh"
#include "G4Box.hh"
#include "G4Cons.hh"
#include "G4Trd.hh"
#include "G4Polyhedra.hh"
#include "G4Element.hh"
#include "G4ElementTable.hh"
#include "G4LogicalVolume.hh"
#include "G4PVPlacement.hh"
#include "G4RotationMatrix.hh"
#include "G4Transform3D.hh"
#include "G4VisAttributes.hh"
#include "G4Colour.hh"

using namespace CLHEP;

DetectorConstruction::DetectorConstruction()
{ }

DetectorConstruction::~DetectorConstruction()
{ }

G4VPhysicalVolume* DetectorConstruction::Construct()
{

  ///////////////////////////////////////////////////////
  //----------------------------------------------------
  // Materials definitions
  //----------------------------------------------------
  ///////////////////////////////////////////////////////
 
  G4String name, symbol;             //a=mass of a mole;
  G4double a, z, density;            //z=mean number of protons;  
  
  G4int ncomponents, natoms;
  G4double temperature, pressure;

  ////////////////////////////////////////////////////////////
  //
  //  Vacuum
  //
  ////////////////////////////////////////////////////////////


  pressure    = 3.e-18*pascal;
  temperature = 2.73*kelvin;
  density     = 1.e-25*g/cm3;

  G4Material* Vacuum = new G4Material(name="Vacuum", z=1., a=1.01*g/mole, 
				      density,kStateGas,temperature,pressure);

    
    ////////////////////////////////////////////////////////////
    //
    // define simple materials
    //
    ////////////////////////////////////////////////////////////
    
    // Aluminum
    density = 2.700*g/cm3;
    a = 26.98*g/mole;
    G4Material* Alum = new G4Material(name="Aluminium", z=13., a, density);
    
    // Germanium
    density = 5.32*g/cm3;
    a = 72.61*g/mole;
    G4Material* Germ = new G4Material(name="Germanium", z=32., a, density);
    
    // Berillyum
    density = 1.848*g/cm3;
    a = 9.012182*g/mole;
    G4Material* Beri = new G4Material(name="Berillyum", z=4., a, density);
    
    // Copper
    density = 8.96*g/cm3;
    a = 63.54*g/mole;
    G4Material* Copp = new G4Material(name="Copper", z=29., a, density);

  
    ////////////////////////////////////////////////////////////
    //
    // define Elements
    //
    ////////////////////////////////////////////////////////////
    
    // Na
    G4Element* Na  = new G4Element(name="Sodium"  ,symbol="Na" , z=11., a=22.990*g/mole);
    
    // I
    G4Element* I = new G4Element(name="Iodine",symbol="I" , z=53., a=126.904*g/mole);
    
    // C
    G4Element* C = new G4Element(name="Carbon",symbol="C" , z=6., a=12.0112*g/mole);
    
    // H
    G4Element* H = new G4Element(name="Hidrogen",symbol="H" , z=1., a=1.00794*g/mole);
    
    // Si
    G4Element* Si = new G4Element(name="Silicon",symbol="Si" , z=14., a=28.0855*g/mole);
    
    // Al
    G4Element* Al = new G4Element(name="Aluminum",symbol="Al" , z=13., a=26.98*g/mole);
    
    // Ba
    G4Element* Ba = new G4Element(name="Barium",symbol="Ba" , z=56., a=137.327*g/mole);
    
    // F
    G4Element* F = new G4Element(name="Fluorine",symbol="F" , z=9., a=18.998*g/mole);

    
    ////////////////////////////////////////////////////////////
    //
    // define materials from elements
    //
    ////////////////////////////////////////////////////////////
    
    // Sodium Iodide
    density = 3.67*g/cm3;
    G4Material* NaI = new G4Material(name="NaI", density, ncomponents=2);
    NaI->AddElement(Na, natoms=1);
    NaI->AddElement(I, natoms=1);
    
    
    //Plastic Scintillator
    density = 1.032*g/cm3;
    G4Material* Plas = new G4Material(name="Plas", density, ncomponents=2);
    Plas->AddElement(C, natoms=9);
    Plas->AddElement(H, natoms=9);
    
    // BaF2
    density = 4.89*g/cm3;
    G4Material* BaF2 = new G4Material(name="BaF2", density, ncomponents=2);
    BaF2->AddElement(Ba, natoms=1);
    BaF2->AddElement(F, natoms=2);
    
    // PTFE
    density = 2.0*g/cm3;
    G4Material* PTFE = new G4Material(name="PTFE", density, ncomponents=2);
    PTFE->AddElement(C, natoms=2);
    PTFE->AddElement(F, natoms=4);
    
    
    /////////////////////////////////////////////////////////
    //------------------------------------------------------
    // Detector geometry
    //------------------------------------------------------
    /////////////////////////////////////////////////////////
    
    /////////////////////////////////////////////////////////////////////
    //
    // World
    //
    /////////////////////////////////////////////////////////////////////
    
    G4double WorldSize= 200.*cm;
    
    G4Box*
    SolidWorld = new G4Box("World",                               //its name
                           WorldSize/2,WorldSize/2,WorldSize/2);  //its size
    
    G4LogicalVolume*
    LogicWorld = new G4LogicalVolume(SolidWorld,      //its solid
                                     Vacuum,	      //its material
                                     "World");        //its name
  
   
    G4VPhysicalVolume*
    PhysiWorld = new G4PVPlacement(0,               //no rotation
                                   G4ThreeVector(),	//at (0,0,0)
                                   "World",         //its name
                                   LogicWorld,		//its logical volume
                                   NULL,            //its mother  volume
                                   false,	       	//no boolean operation
                                   0);              //copy number
  
  ///////////////////////////////////////////////////////////////////////////////// 
  //
  // BaF2 Crystals and reflector
  //   
  /////////////////////////////////////////////////////////////////////////////////

    // Exact geometry as polyhedra with 1 side per segment + a trapezoid
    
    G4int NoOfCrystals=12;
    G4double length_sep = 6.3275*cm;

    // Mother volume for crystals as TUBS
    G4double minRadiusCrystalMother = 2.58*cm;
    G4double maxRadiusCrystalMother = 12.85*cm;
    G4double halfLengthCrystalMother = length_sep;
    G4double startPhiCrystalMother = -30.*degree;
    G4double deltaPhiCrystalMother = 60.*degree;
    G4ThreeVector positionCrystalMother;
    G4RotationMatrix* rotationCrystalMother;
    G4VPhysicalVolume* physiCrystalMother;
    
    G4Tubs* solidCrystalMother = new G4Tubs("CrystalMother",
                                            minRadiusCrystalMother,
                                            maxRadiusCrystalMother,
                                            halfLengthCrystalMother,
                                            startPhiCrystalMother,
                                            deltaPhiCrystalMother);
    
    
    G4LogicalVolume* logicCrystalMother = new G4LogicalVolume(solidCrystalMother,
                                                              Vacuum,
                                                              "CrystalMother");
    
    for(G4int i=0; i<NoOfCrystals; i++){
        G4double  Angle=i*60.*degree;
        G4double X=0.*cm;
        G4double Y=0.*cm;
        G4double Z=-(length_sep);
        
        if(i>5) Z=-Z;
        positionCrystalMother=G4ThreeVector(X,Y,Z);
        rotationCrystalMother=new G4RotationMatrix();
        rotationCrystalMother->rotateZ(Angle);
        physiCrystalMother = new G4PVPlacement(rotationCrystalMother,
                                               positionCrystalMother,
                                               "CrystalMother",logicCrystalMother,
                                               PhysiWorld,false,i);
    }
    
    G4double zPlReflector[2] = {-length_sep,+length_sep};
    G4double rInReflector[2] = {2.58*cm,2.58*cm};
    G4double rOuReflector[2] = {10.825*cm,10.825*cm};
    
    G4ThreeVector positionReflector;
    
    G4Polyhedra* solidReflector
    = new G4Polyhedra("Reflector",-30.*degree,60.*degree,1,2,
                      zPlReflector,rInReflector,rOuReflector);
    
    G4LogicalVolume* logicReflector = new G4LogicalVolume(solidReflector,PTFE,
                                                          "Reflector");
    
    positionReflector=G4ThreeVector(0.0*cm,0.0*cm,0.0*cm);
    G4VPhysicalVolume* physiReflector = new G4PVPlacement(0,positionReflector,
                                                          "Reflector",logicReflector,
                                                          physiCrystalMother,false,0);
    
    G4double zPlCrystal[2] = {-6.25*cm,+6.25*cm};
    G4double rInCrystal[2] = {2.5075*cm,2.5075*cm};
    G4double rOuCrystal[2] = {10.68*cm,10.68*cm};
    
    G4Polyhedra* solidCrystal
    = new G4Polyhedra("Crystal",-30.*degree,60.*degree,1,2,
                      zPlCrystal,rInCrystal,rOuCrystal);
    
    G4LogicalVolume* logicCrystal = new G4LogicalVolume(solidCrystal,BaF2,
                                                        "Crystal");
    
    G4ThreeVector positionCrystal=G4ThreeVector(0.15*cm,0.*cm,0.0*cm);
    G4VPhysicalVolume* physiCrystal = new G4PVPlacement(0,positionCrystal,
                                                        "Crystal",logicCrystal,
                                                        physiReflector,false,0);
    
    G4double halfDXminZ = 6.25*cm;
    G4double halfDXpluZ = 6.25*cm;
    G4double halfDYminZ = 6.25*cm;
    G4double halfDYpluZ = 0.01*cm;
    G4double halfDZ = 0.837*cm;
    G4Trd* solidCrystal2 = new G4Trd("Crystal2",
                                     halfDXminZ,halfDXpluZ,halfDYminZ,halfDYpluZ,halfDZ);
    G4LogicalVolume* logicCrystal2 = new G4LogicalVolume(solidCrystal2,BaF2,
                                                         "Crystal2");
    
    G4RotationMatrix* rotationCrystal2 = new G4RotationMatrix();
    rotationCrystal2->rotateY(270.*degree);
    G4double X=11.662*cm;   
    G4double Y=0.0*cm;
    G4double Z=0.0*cm;
    G4ThreeVector positionCrystal2=G4ThreeVector(X,Y,Z);
    G4VPhysicalVolume* physiCrystal2 = new G4PVPlacement(rotationCrystal2
                                                         ,positionCrystal2,
                                                         "Crystal2",logicCrystal2,
                                                         physiCrystalMother,false,0);


 ////////////////////////////////////////////////////////////////////////////
 //
 // Visualization attributes
 //
 ////////////////////////////////////////////////////////////////////////////


//Invisible vacuum volumes
 LogicWorld->SetVisAttributes (G4VisAttributes::Invisible);
//  LogicVh12->SetVisAttributes (G4VisAttributes::Invisible);
//  LogicSour->SetVisAttributes (G4VisAttributes::Invisible);

 //Green color for reflector
 G4VisAttributes* Att1= new G4VisAttributes(G4Colour(0.0,1.0,0.0));
 Att1->SetVisibility(true);
 logicReflector->SetVisAttributes(Att1);

 //Grey color for Aluminum volumes
 G4VisAttributes* Att2= new G4VisAttributes(G4Colour(0.5,0.5,0.5));
 Att2->SetVisibility(true);
 logicCrystalMother->SetVisAttributes(Att2);

 //Yellow colour
 G4VisAttributes* Att4= new G4VisAttributes(G4Colour(1.0,1.0,0.0));
 Att4->SetVisibility(true);
    logicCrystal->SetVisAttributes(Att4);
    logicCrystal2->SetVisAttributes(Att4);


 //
 //always return the physical World
 //
  
  return PhysiWorld;

}

