///////////////////////////////////////////////
//
//  Dec-2017 E. Nacher --> DetectorConstruction.cc
//
//  Simple geometry made of a NaI crystal
//  and a plastic scint beta detector.
//
///////////////////////////////////////////////


#include "DetectorConstruction.hh"

#include "G4Material.hh"
#include "G4Tubs.hh"
#include "G4Box.hh"
#include "G4Cons.hh"
#include "G4TRD.hh"
#include "G4Polyhedra.hh"
#include "G4Element.hh"
#include "G4ElementTable.hh"
#include "G4LogicalVolume.hh"
#include "G4PVPlacement.hh"
#include "G4RotationMatrix.hh"
#include "G4Transform3D.hh"
#include "G4VisAttributes.hh"
#include "G4Colour.hh"

using namespace CLHEP;

DetectorConstruction::DetectorConstruction()
{ }

DetectorConstruction::~DetectorConstruction()
{ }

G4VPhysicalVolume* DetectorConstruction::Construct()
{

  ///////////////////////////////////////////////////////
  //----------------------------------------------------
  // Materials definitions
  //----------------------------------------------------
  ///////////////////////////////////////////////////////
 
  G4String name, symbol;             //a=mass of a mole;
  G4double a, z, density;            //z=mean number of protons;  
  
  G4int ncomponents, natoms;
  G4double temperature, pressure;

  ////////////////////////////////////////////////////////////
  //
  //  Vacuum
  //
  ////////////////////////////////////////////////////////////


  pressure    = 3.e-18*pascal;
  temperature = 2.73*kelvin;
  density     = 1.e-25*g/cm3;

  G4Material* Vacuum = new G4Material(name="Vacuum", z=1., a=1.01*g/mole, 
				      density,kStateGas,temperature,pressure);

    
    ////////////////////////////////////////////////////////////
    //
    // define simple materials
    //
    ////////////////////////////////////////////////////////////
    
    // Aluminum
    density = 2.700*g/cm3;
    a = 26.98*g/mole;
    G4Material* Alum = new G4Material(name="Aluminium", z=13., a, density);
    
    // Germanium
    density = 5.32*g/cm3;
    a = 72.61*g/mole;
    G4Material* Germ = new G4Material(name="Germanium", z=32., a, density);
    
    // Berillyum
    density = 1.848*g/cm3;
    a = 9.012182*g/mole;
    G4Material* Beri = new G4Material(name="Berillyum", z=4., a, density);
    
    // Copper
    density = 8.96*g/cm3;
    a = 63.54*g/mole;
    G4Material* Copp = new G4Material(name="Copper", z=29., a, density);

  
    ////////////////////////////////////////////////////////////
    //
    // define Elements
    //
    ////////////////////////////////////////////////////////////
    
    // Na
    G4Element* Na  = new G4Element(name="Sodium"  ,symbol="Na" , z=11., a=22.990*g/mole);
    
    // I
    G4Element* I = new G4Element(name="Iodine",symbol="I" , z=53., a=126.904*g/mole);
    
    // C
    G4Element* C = new G4Element(name="Carbon",symbol="C" , z=6., a=12.0112*g/mole);
    
    // H
    G4Element* H = new G4Element(name="Hidrogen",symbol="H" , z=1., a=1.00794*g/mole);
    
    // Si
    G4Element* Si = new G4Element(name="Silicon",symbol="Si" , z=14., a=28.0855*g/mole);
    
    // Al
    G4Element* Al = new G4Element(name="Aluminum",symbol="Al" , z=13., a=26.98*g/mole);
    
    // Ba
    G4Element* Ba = new G4Element(name="Barium",symbol="Ba" , z=56., a=137.327*g/mole);
    
    // F
    G4Element* F = new G4Element(name="Fluorine",symbol="F" , z=9., a=18.998*g/mole);

    
    ////////////////////////////////////////////////////////////
    //
    // define materials from elements
    //
    ////////////////////////////////////////////////////////////
    
    // Sodium Iodide
    density = 3.67*g/cm3;
    G4Material* NaI = new G4Material(name="NaI", density, ncomponents=2);
    NaI->AddElement(Na, natoms=1);
    NaI->AddElement(I, natoms=1);
    
    
    //Plastic Scintillator
    density = 1.032*g/cm3;
    G4Material* Plas = new G4Material(name="Plas", density, ncomponents=2);
    Plas->AddElement(C, natoms=9);
    Plas->AddElement(H, natoms=9);
    
    // BaF2
    density = 4.89*g/cm3;
    G4Material* BaF2 = new G4Material(name="BaF2", density, ncomponents=2);
    BaF2->AddElement(Ba, natoms=1);
    BaF2->AddElement(F, natoms=2);
    
    // PTFE
    density = 2.0*g/cm3;
    G4Material* PTFE = new G4Material(name="PTFE", density, ncomponents=2);
    PTFE->AddElement(C, natoms=2);
    PTFE->AddElement(F, natoms=4);
    
    
    /////////////////////////////////////////////////////////
    //------------------------------------------------------
    // Detector geometry
    //------------------------------------------------------
    /////////////////////////////////////////////////////////
    
    //
    // World
    //
    
    G4double WorldSize= 200.*cm;
    
    G4Box*
    solidWorld = new G4Box("World",		       	       //its name
                           WorldSize/2,WorldSize/2,WorldSize/2);  //its size
    
    G4LogicalVolume*
    logicWorld = new G4LogicalVolume(solidWorld,      	//its solid
                                     Vacuum,	        //its material
                                     "World");		    //its name
    
    
    G4VPhysicalVolume *physWorld = new G4PVPlacement(0,			        //no rotation
                                                     G4ThreeVector(),	//at (0,0,0)
                                                     "World",		    //its name
                                                     logicWorld,		//its logical volume
                                                     NULL,		        //its mother  volume
                                                     false,	       	    //no boolean operation
                                                     0);			    //copy number
  
  //
  // NaI Crystal
  //

    G4double diameter = 10.*cm, thickness = 20.*cm;
    
    G4Tubs *solidNaI = new G4Tubs("solidNaI", 0., diameter/2., thickness/2, 0., 360.*deg);
    
    G4LogicalVolume *logicNaI = new G4LogicalVolume(solidNaI, NaI, "logicNaI");
    
    new G4PVPlacement(0,G4ThreeVector(0.*cm,0.*cm,20.*cm),"physNaI",logicNaI,physWorld,false,0);
    
    //
    // Plastic beta detector
    //
    
    diameter = 6.*cm; thickness = 0.5*cm;
    
    G4Tubs *solidPlas = new G4Tubs("solidPlas", 0., diameter/2., thickness/2, 0., 360.*deg);
    
    G4LogicalVolume *logicPlas = new G4LogicalVolume(solidPlas, Plas, "logicPlas");
    
    new G4PVPlacement(0,G4ThreeVector(0.*cm,0.*cm,-2.*cm),"physPlas",logicPlas,physWorld,false,0);
    
    //
    // source capsule
    //
    
    diameter = 2.*mm; thickness = 50.*um;
    
    G4Tubs *solidCaps = new G4Tubs("solidCaps", 0., diameter/2., thickness/2, 0., 360.*deg);
    
    G4LogicalVolume *logicCaps = new G4LogicalVolume(solidCaps, Plas, "logicCaps");
    
    new G4PVPlacement(0,G4ThreeVector(0.*cm,0.*cm,0.*cm),"physCaps",logicCaps,physWorld,false,0);


 ////////////////////////////////////////////////////////////////////////////
 //
 // Visualization attributes
 //
 ////////////////////////////////////////////////////////////////////////////


//Invisible vacuum volumes
 logicWorld->SetVisAttributes (G4VisAttributes::Invisible);

 //Orange color for NaI
 G4VisAttributes* Att1= new G4VisAttributes(G4Colour(0.42,0.34,0.239));
 Att1->SetVisibility(true);
 logicNaI->SetVisAttributes(Att1);

 //Grey color for Aluminum volumes
 G4VisAttributes* Att2= new G4VisAttributes(G4Colour(0.5,0.5,0.5));
 Att2->SetVisibility(true);
// logicCrystalMother->SetVisAttributes(Att2);

 //Yellow colour for plastic detector
 G4VisAttributes* Att4= new G4VisAttributes(G4Colour(1.,1.,0));
 Att4->SetVisibility(true);
 logicPlas->SetVisAttributes(Att4);


 //
 //always return the physical World
 //
  
  return physWorld;

}

